import React, { useEffect, useState } from 'react';
import {Text, View, ScrollView, Image, StyleSheet, ActivityIndicator, TouchableOpacity} from 'react-native';
import ProtectedRoute from "../components/ProtectedRoute";
import useAuth from "../hooks/useAuth";
import { EXPO_PUBLIC_API_URL } from "../config";
import { usePanier } from './store';
import {useRoute} from "@react-navigation/native";

const HistoriqueCommande = () => {

    const { token, loading: isAuthLoading } = useAuth();
    const [commandes, setCommandes] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [isAdded, setIsAdded] = useState(false);

    const handlePress = async () => {
        if (!DerniereCommandeData || DerniereCommandeData.length === 0) return;

        const articlesACommander = DerniereCommandeData.filter(
            article => article.date_commande === DerniereCommandeData[0].date_commande
        );

        for (const article of articlesACommander) {
            // On utilise la reference (ou la designation en secours) comme ID unique
            const uniqueId = article.reference ;


            ajouterAuPanier({
                id: uniqueId,
                designation: article.designation,
                prix: article.prix_unitaire_HT,
                imageUrl: article.imageUrl,
                quantite: article.quantite_demandee,
            });
        }

        setIsAdded(true);
        setTimeout(() => setIsAdded(false), 2000);
    };
    useEffect(() => {
        const fetchHistorique = async () => {
            try {
                setLoading(true);
                const response = await fetch(`${EXPO_PUBLIC_API_URL}/profil/historique`, {
                    headers: { 'Authorization': `Bearer ${token}` }
                });

                if (!response.ok) throw new Error("Erreur réseau");

                const data = await response.json();

                // Groupement des articles par numéro de commande
                const grouped = data.reduce((acc, curr) => {
                    const id = curr.numero_commande;
                    if (!acc[id]) acc[id] = [];
                    acc[id].push(curr);
                    return acc;
                }, {});

                // On transforme l'objet en tableau de tableaux
                setCommandes(Object.values(grouped));
            } catch (err) {
                setError("Impossible de charger l'historique.");
            } finally {
                setLoading(false);
            }
        };

        if (token) fetchHistorique();
    }, [token]);

    if (loading || isAuthLoading) {
        return (
            <View style={styles.center}><ActivityIndicator size="large" color="#1E3C72" /></View>
        );
    }

    return (
        <ProtectedRoute>
            <ScrollView style={styles.container}>
                <Text style={styles.mainTitle}>Mon Historique</Text>

                {commandes.length === 0 ? (
                    <Text style={styles.emptyText}>Aucune commande passée pour le moment.</Text>
                ) : (
                    commandes.map((commande, index) => (
                        <View key={index} style={styles.orderCard}>
                            <View style={styles.header}>
                                <Text style={styles.dateText}>
                                    Le {new Date(commande[0].date_commande).toLocaleDateString('fr-FR')}
                                </Text>
                                <Text style={styles.refText}>Réf: #{commande[0].numero_commande}</Text>
                            </View>

                            {commande.map((item, idx) => (
                                <View key={idx} style={styles.itemRow}>
                                    <Image
                                        source={{ uri: `${EXPO_PUBLIC_API_URL}/images/produits/${item.imageUrl}` }}
                                        style={styles.itemImage}
                                    />
                                    <View style={{ flex: 1 }}>
                                        <Text style={styles.itemName}>{item.designation}</Text>
                                        <Text style={styles.itemDetails}>Quantité: {item.quantite_demandee}</Text>
                                    </View>
                                    <Text style={styles.itemPrice}>{item.total_ligne_HT} €</Text>
                                </View>
                            ))}

                            <View style={styles.footer}>
                                <Text style={styles.totalLabel}>Total TTC : </Text>
                                <Text style={styles.totalValue}>{commande[0].total_commande_TTC.toFixed(2)} €</Text>
                            </View>

                            <TouchableOpacity
                                style={[
                                    styles.btnAjouter,
                                    { backgroundColor: isAdded ? '#28a745' : '#007bff', paddingHorizontal: 20 }
                                ]}
                                onPress={handlePress}
                                disabled={isAdded}
                            >
                                <Text style={styles.btnAjouterText}>
                                    {isAdded ? "Ajouté ! ✅" : "Re-commander"}
                                </Text>
                            </TouchableOpacity>
                        </View>
                    ))
                )}
            </ScrollView>
        </ProtectedRoute>
    );
};

const styles = StyleSheet.create({
    container: { flex: 1, backgroundColor: '#F4F7F9', padding: 15 },
    center: { flex: 1, justifyContent: 'center', alignItems: 'center' },
    mainTitle: { fontSize: 24, fontWeight: '800', marginBottom: 20, color: '#1A1A1A' },
    emptyText: { textAlign: 'center', marginTop: 40, color: '#666' },
    orderCard: {
        backgroundColor: '#FFF',
        borderRadius: 15,
        padding: 15,
        marginBottom: 20,
        elevation: 4,
        shadowColor: '#000',
        shadowOpacity: 0.1,
        shadowRadius: 10,
    },
    header: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        borderBottomWidth: 1,
        borderBottomColor: '#EEE',
        paddingBottom: 10,
        marginBottom: 10
    },
    dateText: { fontWeight: 'bold', color: '#333' },
    refText: { color: '#888', fontSize: 12 },
    itemRow: { flexDirection: 'row', alignItems: 'center', marginBottom: 10 },
    itemImage: { width: 40, height: 40, borderRadius: 5, marginRight: 10 },
    itemName: { fontSize: 14, fontWeight: '600' },
    itemDetails: { fontSize: 12, color: '#666' },
    itemPrice: { fontWeight: 'bold' },
    footer: {
        marginTop: 10,
        paddingTop: 10,
        borderTopWidth: 1,
        borderTopColor: '#EEE',
        flexDirection: 'row',
        justifyContent: 'space-between'
    },
    btnAjouter: {
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'center',
        paddingVertical: 6,
        paddingHorizontal: 12,
        borderRadius: 8, // Arrondi plus subtil
        alignSelf: 'center',

    },

    btnAjouterText: {
        color: '#ffffff',
        fontSize: 15,
        fontWeight: '600', // Moins épais que 'bold' pour plus de finesse
        // On retire l'uppercase et le letterSpacing qui déforment l'écriture
        textTransform: 'none',
        letterSpacing: 0.5,
    },
    totalLabel: { fontSize: 16, fontWeight: '700' },
    totalValue: { fontSize: 18, fontWeight: '900', color: '#1E3C72' }
});

export default HistoriqueCommande;