import BaseClient from "./BaseClient";
import { Callback, ClientOptions, DefaultResponse, FilteringParameters } from "./models/index";
import { Bounce, BounceActivationResponse, BounceCounts, BounceDump, BounceFilteringParameters, Bounces, BrowserUsageCounts, ClickCounts, ClickLocationCounts, ClickPlatformUsageCounts, CreateInboundRuleRequest, CreateMessageStreamRequest, CreateSuppressionsRequest, CreateTemplateRequest, CreateWebhookRequest, DeleteSuppressionsRequest, DeliveryStatistics, EmailClientUsageCounts, EmailPlaformUsageCounts, EmailReadTimesCounts, InboundMessageDetails, InboundMessages, InboundMessagesFilteringParameters, InboundRule, InboundRules, Message, MessageSendingResponse, MessageStream, MessageStreamArchiveResponse, MessageStreams, MessageStreamsFilteringParameters, MessageStreamUnarchiveResponse, OpenCounts, OutboundMessageClicks, OutboundMessageClicksFilteringParameters, OutboundMessageDetails, OutboundMessageDump, OutboundMessageOpens, OutboundMessageOpensFilteringParameters, OutboundMessages, OutboundMessagesFilteringParameters, OutboundStatistics, SentCounts, Server, SpamCounts, StatisticsFilteringParameters, Suppressions, SuppressionFilteringParameters, SuppressionStatuses, Template, TemplatedMessage, TemplateFilteringParameters, Templates, TemplateValidation, TemplateValidationOptions, TrackedEmailCounts, UpdateMessageStreamRequest, UpdateServerRequest, UpdateTemplateRequest, UpdateWebhookRequest, Webhook, WebhookFilteringParameters, Webhooks } from "./models/index";
/**
 * Server client class that can be used to interact with an individual Postmark Server.
 */
export default class ServerClient extends BaseClient {
    /**
     * Create a client.
     *
     * @param serverToken - The token for the server that you wish to interact with.
     * @param configOptions - Options to customize the behavior of the this client.
     */
    constructor(serverToken: string, configOptions?: ClientOptions.Configuration);
    /** Send a single email message.
     *
     * @param email - Email message to send.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    sendEmail(email: Message, callback?: Callback<MessageSendingResponse>): Promise<MessageSendingResponse>;
    /**
     * Send a batch of email messages.
     *
     * @param emails - An array of messages to send.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    sendEmailBatch(emails: Message[], callback?: Callback<MessageSendingResponse[]>): Promise<MessageSendingResponse[]>;
    /**
     * Send a message using a template.
     *
     * @param template - Message you wish to send.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    sendEmailWithTemplate(template: TemplatedMessage, callback?: Callback<MessageSendingResponse>): Promise<MessageSendingResponse>;
    /**
     * Send a batch of template email messages.
     *
     * @param templates - An array of templated messages you wish to send using this Client.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    sendEmailBatchWithTemplates(templates: TemplatedMessage[], callback?: Callback<MessageSendingResponse[]>): Promise<MessageSendingResponse[]>;
    /**
     * Get bounce statistic information for the associated Server.
     *
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getDeliveryStatistics(callback?: Callback<DeliveryStatistics>): Promise<DeliveryStatistics>;
    /**
     * Get a batch of bounces.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getBounces(filter?: BounceFilteringParameters, callback?: Callback<Bounces>): Promise<Bounces>;
    /**
     * Get details for a specific Bounce.
     *
     * @param id - The ID of the Bounce you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getBounce(id: number, callback?: Callback<Bounce>): Promise<Bounce>;
    /**
     * Get a Bounce Dump for a specific Bounce.
     *
     * @param id - The ID of the Bounce for which you wish to retrieve Bounce Dump.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getBounceDump(id: number, callback?: Callback<BounceDump>): Promise<BounceDump>;
    /**
     * Activate email address that was deactivated due to a Bounce.
     *
     * @param id - The ID of the Bounce for which you wish to activate the associated email.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    activateBounce(id: number, callback?: Callback<BounceActivationResponse>): Promise<BounceActivationResponse>;
    /**
     * Get the list of templates associated with this server.
     *
     * @param filter - Optional filtering options.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getTemplates(filter?: TemplateFilteringParameters, callback?: Callback<Templates>): Promise<Templates>;
    /**
     * Get the a specific template associated with this server.
     *
     * @param idOrAlias - ID or alias for the template you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getTemplate(idOrAlias: (number | string), callback?: Callback<Template>): Promise<Template>;
    /**
     * Delete a template associated with this server.
     *
     * @param idOrAlias - ID or template alias you wish to delete.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteTemplate(idOrAlias: (number | string), callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Create a new template on the associated server.
     *
     * @param options - Configuration options to be used to create the Template.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createTemplate(options: CreateTemplateRequest, callback?: Callback<Template>): Promise<Template>;
    /**
     * Update a template on the associated server.
     *
     * @param idOrAlias - Id or alias of the template you wish to update.
     * @param options - Template options you wish to update.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editTemplate(idOrAlias: (number | string), options: UpdateTemplateRequest, callback?: Callback<Template>): Promise<Template>;
    /**
     * Validate template markup to verify that it will be parsed. Also provides a recommended template
     * model to be used when sending using the specified template content.
     *
     * @param options - The template content you wish to validate.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    validateTemplate(options: TemplateValidationOptions, callback?: Callback<TemplateValidation>): Promise<TemplateValidation>;
    /**
     * Get the information for the Server associated with this Client.
     *
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getServer(callback?: Callback<Server>): Promise<Server>;
    /**
     * Modify the Server associated with this Client.
     *
     * @param options - The options you wish to modify.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editServer(options: UpdateServerRequest, callback?: Callback<Server>): Promise<Server>;
    /**
     * Get a batch of Outbound Messages.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getOutboundMessages(filter?: OutboundMessagesFilteringParameters, callback?: Callback<OutboundMessages>): Promise<OutboundMessages>;
    /**
     * Get details for a specific Outbound Message.
     *
     * @param messageId - The ID of the OutboundMessage you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getOutboundMessageDetails(messageId: string, callback?: Callback<OutboundMessageDetails>): Promise<OutboundMessageDetails>;
    /**
     * Get details for a specific Outbound Message.
     *
     * @param messageId - The ID of the OutboundMessage you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getOutboundMessageDump(messageId: string, callback?: Callback<OutboundMessageDump>): Promise<OutboundMessageDump>;
    /**
     * Get a batch of Inbound Messages.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getInboundMessages(filter?: InboundMessagesFilteringParameters, callback?: Callback<InboundMessages>): Promise<InboundMessages>;
    /**
     * Get details for a specific Inbound Message.
     *
     * @param messageId - The ID of the Inbound Message you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getInboundMessageDetails(messageId: string, callback?: Callback<InboundMessageDetails>): Promise<InboundMessageDetails>;
    /**
     * Cause an Inbound Message to bypass filtering rules defined on this Client's associated Server.
     *
     * @param messageId - The ID of the Inbound Message for which you wish to bypass the filtering rules.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    bypassBlockedInboundMessage(messageId: string, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Request that Postmark retry POSTing to the Inbound Hook for the specified message.
     *
     * @param messageId - The ID of the Inbound Message for which you wish to retry the inbound hook.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    retryInboundHookForMessage(messageId: string, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Get the Opens for Outbound Messages.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageOpens(filter?: OutboundMessageOpensFilteringParameters, callback?: Callback<OutboundMessageOpens>): Promise<OutboundMessageOpens>;
    /**
     * Get details of Opens for specific Outbound Message.
     *
     * @param messageId - Message ID of the message for which you wish to retrieve Opens.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageOpensForSingleMessage(messageId: string, filter?: OutboundMessageOpensFilteringParameters, callback?: Callback<OutboundMessageOpens>): Promise<OutboundMessageOpens>;
    /**
     * Get the Clicks for Outbound Messages.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageClicks(filter?: OutboundMessageClicksFilteringParameters, callback?: Callback<OutboundMessageClicks>): Promise<OutboundMessageClicks>;
    /**
     * Get Click information for a single Outbound Message.
     *
     * @param messageId - The MessageID for which clicks should be retrieved.
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageClicksForSingleMessage(messageId: string, filter?: OutboundMessageClicksFilteringParameters, callback?: Callback<OutboundMessageClicks>): Promise<OutboundMessageClicks>;
    /**
     * Get overview statistics on Outbound Messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getOutboundOverview(filter?: StatisticsFilteringParameters, callback?: Callback<OutboundStatistics>): Promise<OutboundStatistics>;
    /**
     * Get statistics on email sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getSentCounts(filter?: StatisticsFilteringParameters, callback?: Callback<SentCounts>): Promise<SentCounts>;
    /**
     * Get statistiscs on emails that bounced after being sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getBounceCounts(filter?: StatisticsFilteringParameters, callback?: Callback<BounceCounts>): Promise<BounceCounts>;
    /**
     * Get SPAM complaint statistics for email sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getSpamComplaintsCounts(filter?: StatisticsFilteringParameters, callback?: Callback<SpamCounts>): Promise<SpamCounts>;
    /**
     * Get email tracking statistics for messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getTrackedEmailCounts(filter?: StatisticsFilteringParameters, callback?: Callback<TrackedEmailCounts>): Promise<TrackedEmailCounts>;
    /**
     * Get Open statistics for messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getEmailOpenCounts(filter?: StatisticsFilteringParameters, callback?: Callback<OpenCounts>): Promise<OpenCounts>;
    /**
     * Get Email Client Platform statistics for messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getEmailOpenPlatformUsage(filter?: StatisticsFilteringParameters, callback?: Callback<EmailPlaformUsageCounts>): Promise<EmailPlaformUsageCounts>;
    /**
     * Get statistics on which Email Clients were used to open messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getEmailOpenClientUsage(filter?: StatisticsFilteringParameters, callback?: Callback<EmailClientUsageCounts>): Promise<EmailClientUsageCounts>;
    /**
     * Get Read Time statistics for messages sent from the Server associated with this Client.
     * @param filter Optional filtering parameters.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getEmailOpenReadTimes(filter?: StatisticsFilteringParameters, callback?: Callback<EmailReadTimesCounts>): Promise<EmailReadTimesCounts>;
    /**
     * Get total clicks statistics for tracked links for messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getClickCounts(filter?: StatisticsFilteringParameters, callback?: Callback<ClickCounts>): Promise<ClickCounts>;
    /**
     * Get browser family statistics for tracked links for messages sent from the Server associated with this Client.
     * @param filter Optional filtering parameters.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getClickBrowserUsage(filter?: StatisticsFilteringParameters, callback?: Callback<BrowserUsageCounts>): Promise<BrowserUsageCounts>;
    /**
     * Get browser platform statistics for tracked links for messages sent from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getClickPlatformUsage(filter?: StatisticsFilteringParameters, callback?: Callback<ClickPlatformUsageCounts>): Promise<ClickPlatformUsageCounts>;
    /**
     * Get click location (in HTML or Text body of the email) statistics for tracked links for messages sent
     * from the Server associated with this Client.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getClickLocation(filter?: StatisticsFilteringParameters, callback?: Callback<ClickLocationCounts>): Promise<ClickLocationCounts>;
    /**
     * Create an Inbound Rule Trigger.
     *
     * @param options - Configuration options to be used when creating this Trigger.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createInboundRuleTrigger(options: CreateInboundRuleRequest, callback?: Callback<InboundRule>): Promise<InboundRule>;
    /**
     * Delete an Inbound Rule Trigger.
     *
     * @param id - The ID of the Inbound Rule Trigger you wish to delete.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteInboundRuleTrigger(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Get a list of Inbound Rule Triggers.
     *
     * @param filter - Optional filtering parameters.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getInboundRuleTriggers(filter?: FilteringParameters, callback?: Callback<InboundRules>): Promise<InboundRules>;
    /**
     * Get the list of Webhooks for specific server.
     *
     * @param filter - Optional filtering parameters
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getWebhooks(filter?: WebhookFilteringParameters, callback?: Callback<Webhooks>): Promise<Webhooks>;
    /**
     * Get details for a specific Webhook.
     *
     * @param id - The ID of the Webhook you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getWebhook(id: number, callback?: Callback<Webhook>): Promise<Webhook>;
    /**
     * Create a Webhook on the associated server.
     *
     * @param options - Configuration options to be used when creating Webhook trigger.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createWebhook(options: CreateWebhookRequest, callback?: Callback<Webhook>): Promise<Webhook>;
    /**
     * Update Webhook on the associated server.
     *
     * @param id - Id of the webhook you wish to update.
     * @param options - Webhook options you wish to update.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editWebhook(id: number, options: UpdateWebhookRequest, callback?: Callback<Webhook>): Promise<Webhook>;
    /**
     * Delete an existing Webhook.
     *
     * @param id - The ID of the Webhook you wish to delete.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteWebhook(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Get the list of message streams on a server.
     *
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageStreams(filter?: MessageStreamsFilteringParameters, callback?: Callback<MessageStreams>): Promise<MessageStreams>;
    /**
     * Get details for a specific message stream on a server.
     *
     * @param id - The ID of the message stream you wish to retrieve.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getMessageStream(id: string, callback?: Callback<MessageStream>): Promise<MessageStream>;
    /**
     * Update message stream on the associated server.
     *
     * @param id - Id of the webhook you wish to update.
     * @param options - Webhook options you wish to update.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editMessageStream(id: string, options: UpdateMessageStreamRequest, callback?: Callback<MessageStream>): Promise<MessageStream>;
    /**
     * Create a message stream on the associated server.
     *
     * @param options - Configuration options to be used when creating message stream on the server.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createMessageStream(options: CreateMessageStreamRequest, callback?: Callback<MessageStream>): Promise<MessageStream>;
    /**
     * Archive a message stream on the associated server.
     *
     * @param options - Configuration options to be used when creating message stream on the server.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    archiveMessageStream(id: string, callback?: Callback<MessageStreamArchiveResponse>): Promise<MessageStreamArchiveResponse>;
    /**
     * Unarchive a message stream on the associated server.
     *
     * @param options - Configuration options to be used when creating message stream on the server.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    unarchiveMessageStream(id: string, callback?: Callback<MessageStreamUnarchiveResponse>): Promise<MessageStreamUnarchiveResponse>;
    /**
     * Get the list of suppressions for a message stream on a server.
     *
     * @param messageStream - Select message stream
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getSuppressions(messageStream: string, filter?: SuppressionFilteringParameters, callback?: Callback<Suppressions>): Promise<Suppressions>;
    /**
     * Add email addresses to a suppressions list on a message stream on a server.
     *
     * @param messageStream - Select message stream
     * @param options - Suppressions you wish to add.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createSuppressions(messageStream: string, options: CreateSuppressionsRequest, callback?: Callback<SuppressionStatuses>): Promise<SuppressionStatuses>;
    /**
     * Delete email addresses from a suppressions list on a message stream on a server.
     *
     * @param messageStream - Select message stream
     * @param options - Suppressions you wish to delete.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteSuppressions(messageStream: string, options: DeleteSuppressionsRequest, callback?: Callback<SuppressionStatuses>): Promise<SuppressionStatuses>;
}
