"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var ErrorHandler_1 = require("./errors/ErrorHandler");
var HttpClient_1 = require("./HttpClient");
var packageJson = require("../../package.json");
var CLIENT_VERSION = packageJson.version;
/**
 * Base client class from which client classes can be implemented, in our case, AccountClient and ServerClient classes.
 * This class is NOT intended to be instantiated directly.
 */
var BaseClient = /** @class */ (function () {
    function BaseClient(token, authHeader, configOptions) {
        this.errorHandler = new ErrorHandler_1.ErrorHandler();
        this.verifyToken(token);
        this.token = token.trim();
        this.authHeader = authHeader;
        this.clientVersion = CLIENT_VERSION;
        this.httpClient = new HttpClient_1.AxiosHttpClient(configOptions);
    }
    BaseClient.prototype.setClientOptions = function (configOptions) {
        this.httpClient.initHttpClient(configOptions);
    };
    BaseClient.prototype.getClientOptions = function () {
        return this.httpClient.clientOptions;
    };
    /**
     * Process http request with sending body - data.
     *
     * @see processRequest for more details.
     */
    BaseClient.prototype.processRequestWithBody = function (method, path, body, callback) {
        return this.processRequest(method, path, {}, body, callback);
    };
    /**
     * Process http request without sending body - data.
     *
     * @see processRequest for more details.
     */
    BaseClient.prototype.processRequestWithoutBody = function (method, path, queryParameters, callback) {
        if (queryParameters === void 0) { queryParameters = {}; }
        return this.processRequest(method, path, queryParameters, null, callback);
    };
    /**
     * Process request for Postmark ClientOptions.
     *
     * @param method - see processHttpRequest for details
     * @param path - see processHttpRequest for details
     * @param queryParameters - see processHttpRequest for details
     * @param body - see processHttpRequest for details
     * @param callback - callback function to be executed.
     *
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    BaseClient.prototype.processRequest = function (method, path, queryParameters, body, callback) {
        var httpRequest = this.processHttpRequest(method, path, queryParameters, body);
        this.processCallbackRequest(httpRequest, callback);
        return httpRequest;
    };
    /**
     * Process HTTP request.
     *
     * @param method - Which type of http request will be executed.
     * @param path - API URL endpoint.
     * @param queryParameters - Querystring parameters used for http request.
     * @param body - Data sent with http request.
     *
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    BaseClient.prototype.processHttpRequest = function (method, path, queryParameters, body) {
        return this.httpClient.httpRequest(method, path, queryParameters, body, this.getComposedHttpRequestHeaders())
            .then(function (response) { return response; })
            .catch(function (error) { return Promise.reject(error); });
    };
    /**
     * Process callback function for HTTP request.
     *
     * @param httpRequest - HTTP request for which callback will be executed
     * @param callback - callback function to be executed.
     */
    BaseClient.prototype.processCallbackRequest = function (httpRequest, callback) {
        if (callback) {
            httpRequest
                .then(function (response) { return callback(null, response); })
                .catch(function (error) { return callback(error, null); });
        }
    };
    /**
     * JSON object with default headers sent by HTTP request.
     */
    BaseClient.prototype.getComposedHttpRequestHeaders = function () {
        var _a;
        return _a = {},
            _a[this.authHeader] = this.token,
            _a["Accept"] = "application/json",
            _a["Content-Type"] = "application/json",
            _a["User-Agent"] = "Postmark.JS - ".concat(this.clientVersion),
            _a;
    };
    /**
     * Token can't be empty.
     *
     * @param {string} token - HTTP request token
     */
    BaseClient.prototype.verifyToken = function (token) {
        if (!token || token.trim() === "") {
            throw this.errorHandler.buildError("A valid API token must be provided.");
        }
    };
    /**
     * Set default values for count and offset when doing filtering with API requests if they are not specified by filter.
     */
    BaseClient.prototype.setDefaultPaginationValues = function (filter) {
        filter.count = filter.count || 100;
        filter.offset = filter.offset || 0;
    };
    return BaseClient;
}());
exports.default = BaseClient;
//# sourceMappingURL=BaseClient.js.map