import { ErrorHandler } from "./errors/ErrorHandler";
import { Callback, ClientOptions, FilteringParameters, HttpClient } from "./models";
/**
 * Base client class from which client classes can be implemented, in our case, AccountClient and ServerClient classes.
 * This class is NOT intended to be instantiated directly.
 */
export default abstract class BaseClient {
    clientVersion: string;
    httpClient: HttpClient;
    protected errorHandler: ErrorHandler;
    private readonly authHeader;
    private readonly token;
    protected constructor(token: string, authHeader: string, configOptions?: ClientOptions.Configuration);
    setClientOptions(configOptions: ClientOptions.Configuration): void;
    getClientOptions(): ClientOptions.Configuration;
    /**
     * Process http request with sending body - data.
     *
     * @see processRequest for more details.
     */
    protected processRequestWithBody<T>(method: ClientOptions.HttpMethod, path: string, body: (null | object), callback?: Callback<T>): Promise<T>;
    /**
     * Process http request without sending body - data.
     *
     * @see processRequest for more details.
     */
    protected processRequestWithoutBody<T>(method: ClientOptions.HttpMethod, path: string, queryParameters?: object, callback?: Callback<T>): Promise<T>;
    /**
     * Process request for Postmark ClientOptions.
     *
     * @param method - see processHttpRequest for details
     * @param path - see processHttpRequest for details
     * @param queryParameters - see processHttpRequest for details
     * @param body - see processHttpRequest for details
     * @param callback - callback function to be executed.
     *
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    private processRequest;
    /**
     * Process HTTP request.
     *
     * @param method - Which type of http request will be executed.
     * @param path - API URL endpoint.
     * @param queryParameters - Querystring parameters used for http request.
     * @param body - Data sent with http request.
     *
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    private processHttpRequest;
    /**
     * Process callback function for HTTP request.
     *
     * @param httpRequest - HTTP request for which callback will be executed
     * @param callback - callback function to be executed.
     */
    private processCallbackRequest;
    /**
     * JSON object with default headers sent by HTTP request.
     */
    private getComposedHttpRequestHeaders;
    /**
     * Token can't be empty.
     *
     * @param {string} token - HTTP request token
     */
    private verifyToken;
    /**
     * Set default values for count and offset when doing filtering with API requests if they are not specified by filter.
     */
    protected setDefaultPaginationValues(filter: FilteringParameters): void;
}
