import BaseClient from "./BaseClient";
import { Callback, ClientOptions, DefaultResponse, FilteringParameters } from "./models";
import { CreateDomainRequest, CreateServerRequest, CreateSignatureRequest, DomainDetails, Domains, Server, ServerFilteringParameters, Servers, SignatureDetails, Signatures, TemplatesPush, TemplatesPushRequest, UpdateDomainRequest, UpdateServerRequest, UpdateSignatureRequest, DataRemovalRequest, DataRemovalStatus } from "./models";
export default class AccountClient extends BaseClient {
    /**
     * Create a new AccountClient
     * @param accountToken The account token that should be used with requests.
     * @param configOptions Various options to customize client behavior.
     */
    constructor(accountToken: string, configOptions?: ClientOptions.Configuration);
    /**
     * Retrieve a list of Servers.
     *
     * @param filter - An optional filter for which data is retrieved.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getServers(filter?: ServerFilteringParameters, callback?: Callback<Servers>): Promise<Servers>;
    /**
     * Retrieve a single server by ID.
     *
     * @param id - The ID of the Server for which you wish to retrieve details.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getServer(id: number, callback?: Callback<Server>): Promise<Server>;
    /**
     * Create a new Server.
     *
     * @param options - The options to be used to create new Server.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createServer(options: CreateServerRequest, callback?: Callback<Server>): Promise<Server>;
    /**
     * Modify the Server associated with this Client.
     *
     * @param id - The ID of the Server you wish to update.
     * @param options - The options to be used to create new Server.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editServer(id: number, options: UpdateServerRequest, callback?: Callback<Server>): Promise<Server>;
    /**
     * Modify the Server associated with this Client.
     *
     * @param id - The ID of the Domain you wish to delete.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteServer(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Retrieve a batch of Domains.
     *
     * @param filter - An optional filter for which data is retrieved.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getDomains(filter?: FilteringParameters, callback?: Callback<Domains>): Promise<Domains>;
    /**
     * Retrieve a single Domain by ID.
     *
     * @param id - The ID of the Domain for which you wish to retrieve details.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getDomain(id: number, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Create a new Domain.
     *
     * @param options - The options to be used to create new Domain.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createDomain(options: CreateDomainRequest, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Update a Domain.
     *
     * @param id - The ID of the Domain you wish to update.
     * @param domain - The values on the Domain you wish to update.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editDomain(id: number, options: UpdateDomainRequest, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Delete a Domain.
     *
     * @param id - The ID of the Domain you wish to delete.
     * @param options - The options to be used in create Domain.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteDomain(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Trigger Domain DKIM key verification.
     *
     * @param id - The ID of the Domain you wish to trigger DKIM verification for.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    verifyDomainDKIM(id: number, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Trigger Domain DKIM key verification.
     *
     * @param id - The ID of the Domain you wish to trigger DKIM verification for.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    verifyDomainReturnPath(id: number, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Trigger Domain DKIM key verification.
     *
     * @param id - The ID of the Domain you wish to trigger DKIM verification for.
     * @param callback If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    verifyDomainSPF(id: number, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Trigger Domain DKIM key verification.
     *
     * @param id - The ID of the Domain you wish to trigger DKIM verification for.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    rotateDomainDKIM(id: number, callback?: Callback<DomainDetails>): Promise<DomainDetails>;
    /**
     * Retrieve a single Sender Signature by ID.
     *
     * @param id - The ID of the Sender Signature for which you wish to retrieve details.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getSenderSignature(id: number, callback?: Callback<SignatureDetails>): Promise<SignatureDetails>;
    /**
     * Retrieve a batch of Sender Signatures.
     *
     * @param filter - An optional filter for which data is retrieved.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getSenderSignatures(filter?: FilteringParameters, callback?: Callback<Signatures>): Promise<Signatures>;
    /**
     * Create a new Sender Signature.
     *
     * @param options - The options to be used to create new Sender Signature.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    createSenderSignature(options: CreateSignatureRequest, callback?: Callback<SignatureDetails>): Promise<SignatureDetails>;
    /**
     * Update a Sender Signature.
     *
     * @param id - The ID of the Sender Signature for which you wish to update.
     * @param options - The values on the Sender Signature you wish to update.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    editSenderSignature(id: number, options: UpdateSignatureRequest, callback?: Callback<SignatureDetails>): Promise<SignatureDetails>;
    /**
     * Delete a Domain.
     *
     * @param id - The ID of the Domain you wish to delete.
     * @param options - The options to be used in create Domain.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    deleteSenderSignature(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Request a new confirmation email to be sent to the email address associated with a Sender Signature.
     *
     * @param id - The ID of the Sender Signature.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    resendSenderSignatureConfirmation(id: number, callback?: Callback<DefaultResponse>): Promise<DefaultResponse>;
    /**
     * Request that the SPF records for Sender Signature be verified.
     *
     * @param id - The ID of the Sender Signature.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    verifySenderSignatureSPF(id: number, callback?: Callback<SignatureDetails>): Promise<SignatureDetails>;
    /**
     * Request that the SPF records for Sender Signature be verified.
     *
     * @param id - The ID of the Sender Signature.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    requestNewDKIMForSenderSignature(id: number, callback?: Callback<SignatureDetails>): Promise<SignatureDetails>;
    /**
     * Request a push of templates from one server to another.
     *
     * @param options - details for pushing templates from one place to another.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    pushTemplates(options: TemplatesPushRequest, callback?: Callback<TemplatesPush>): Promise<TemplatesPush>;
    /**
     * Request a data removal.
     *
     * @param options - details for creating data removal request
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    requestDataRemoval(options: DataRemovalRequest, callback?: Callback<DataRemovalStatus>): Promise<DataRemovalStatus>;
    /**
     * Retrieve a single data removal status by ID.
     *
     * @param id - The ID of the DataRemoval for which you wish to retrieve details.
     * @param callback - If the callback is provided, it will be passed to the resulting promise as a continuation.
     * @returns A promise that will complete when the API responds (or an error occurs).
     */
    getDataRemovalStatus(id: number, callback?: Callback<DataRemovalStatus>): Promise<DataRemovalStatus>;
}
