<?php

class CommandeControleur{

    /* Espace déclaration des propriétés */
    private $parametres = []; // Remarque : cette proriété est de type => tableau

    private $oModele; // Remarque : cette proriété est de type => objet
    private $oVue; // Remarque : cette proriété est de type => objet

    /* Constructeur */
    /**
     * @param $parametres // contient les valeurs du tableau $_REQUEST ($_GET et $_POST))
     */
    public function __construct($parametres){
        // Initialisation de la propriété parametres
        $this->parametres = $parametres;

        $this->oModele = new CommandeModele($this->parametres);
        $this->oVue    = new CommandeVue($this->parametres);
    }

    /**
     * Détecte la source et retourne un booléen selon la BDD :
     * 0 = Application Vendeur (intranet)
     * 1 = Application Client (mobile)
     */
    private function detectSourceBool(): int
    {
        // 1) Header explicite (si ton app mobile l’envoie)
        if (!empty($_SERVER['HTTP_X_APP_SOURCE'])) {
            $val = strtolower(trim($_SERVER['HTTP_X_APP_SOURCE']));
            if ($val === 'mobile' || $val === 'client') {
                return 1; // Application Client
            }
            if ($val === 'intranet' || $val === 'vendeur') {
                return 0; // Application Vendeur
            }
        }

        // 2) Heuristique User-Agent (okhttp/CFNetwork/Android/iPhone => mobile)
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        if (preg_match('/okhttp|cfnetwork|android|iphone|ipad|dart|flutter/i', $ua)) {
            return 1;
        }

        // 3) Fallback par URL (ton critère simple)
        $host = $_SERVER['HTTP_HOST']    ?? '';
        $uri  = $_SERVER['REQUEST_URI']  ?? '';
        $full = $host . $uri;

        // Si tu es sur l’URL intranet locale spécifiée -> 0, sinon 1
        if (stripos($full, 'localhost/sp-vendeurs/sp-vendeurs-11/index.php') !== false) {
            return 0; // Vendeur
        }

        return 1; // par défaut, Application Client (mobile)
    }

    public function charger_panier()
    {
        $json = file_get_contents('php://input');

        if ($json) {
            $_SESSION['panier_json'] = $json;
            echo "✅ Panier stocké";
        } else {
            http_response_code(400);
            echo "❌ JSON vide";
        }
    }

    public function lister(){
        $lesCommandes = $this->oModele->getListeCommandes();
        $this->oVue->genererAffichageListe($lesCommandes);
    }

    public function form_consulter()
    {
        $articles = [];

        if (isset($_SESSION['panier_json'])) {
            $json = json_decode($_SESSION['panier_json'], true);

            foreach ($json as $ref => $quantite) {
                $produit = $this->oModele->getProduitByReference($ref);
                if ($produit) {
                    $articles[] = [
                        'reference' => $produit['reference'],
                        'designation' => $produit['designation'],
                        'quantite' => $quantite,
                        'tarif' => $produit['prix_unitaire_HT'],
                        'montantClient' => round($produit['prix_unitaire_HT'] * 1.36, 2)
                    ];
                }
            }
        }

        // 💥 assignation pour la vue
        $this->oVue->afficherPanierDepuisFormConsulter($articles);
    }

    public function form_ajouter(){
        $prepareCommande = new CommandeTable();
        $this->oVue->genererAffichageFiche($prepareCommande);
    }

    public function ajouter(){
        // contrôle des données
        $controlDonnes = new CommandeTable($this->parametres);

        if($controlDonnes->getAutorisationBD() == false){
            $this->oVue->genererAffichageFiche($controlDonnes);
        }else{
            $this->oModele->addCommande($controlDonnes);
            $this->lister();
        }
    }

    public function form_supprimer()
    {
        if (isset($_POST['reference']) && isset($_SESSION['panier_json'])) {
            $panier = json_decode($_SESSION['panier_json'], true);

            unset($panier[$_POST['reference']]); // supprime la ligne
            $_SESSION['panier_json'] = json_encode($panier); // remplace

            header("Location: index.php?gestion=commande&action=form_consulter");
            exit;
        }
    }

    public function form_modifier()
    {
        if (isset($_POST['reference'], $_POST['nouvelle_quantite'], $_SESSION['panier_json'])) {
            $ref = $_POST['reference'];
            $qte = max(1, intval($_POST['nouvelle_quantite']));

            $panier = json_decode($_SESSION['panier_json'], true);

            if (isset($panier[$ref])) {
                $panier[$ref] = $qte;
                $_SESSION['panier_json'] = json_encode($panier);
            }
        }

        // Redirige vers le panier mis à jour
        header("Location: index.php?gestion=commande&action=form_consulter");
        exit;
    }

    public function modifier(){
        // contrôle des données
        $controlDonnes = new CommandeTable($this->parametres);

        if($controlDonnes->getAutorisationBD() == false){
            $this->oVue->genererAffichageFiche($controlDonnes);
        }else{
            $this->oModele->editCommande($controlDonnes);
            $this->lister();
        }
    }

    public function supprimer(){
        // contrôle : client possède-t-il une commande ?
        $controleSuppression = $this->oModele->supressionPossible();

        if($controleSuppression == false){
            CommandeTable::setMessageErreur("Impossible de supprimer le client. Celui ci possède au moins une commande.");
            $this->oVue->genererAffichageFiche($this->oModele->getUnCommande());
        }else{
            $this->oModele->deleteCommande();
            $this->lister();
        }
    }

    public function form_valider_commande()
    {
        if (!isset($_SESSION['panier_json'])) {
            header("Location: index.php?gestion=commande&action=form_valider_commande");
            exit;
        }

        $panier = json_decode($_SESSION['panier_json'], true);
        $articles = [];

        $totalHT = 0;

        foreach ($panier as $reference => $quantite) {
            $produit = $this->oModele->getProduitByReference($reference);
            if ($produit) {
                $totalLigne = $produit['prix_unitaire_HT'] * $quantite;
                $articles[] = [
                    'reference' => $reference,
                    'designation' => $produit['designation'],
                    'quantite' => $quantite,
                    'prix_ht' => $produit['prix_unitaire_HT'],
                    'total_ligne' => $totalLigne
                ];
                $totalHT += $totalLigne;
            }
        }

        $tva = $totalHT * 0.2;
        $dateCommande = date("Y-m-d");
        $dateLivraison = date("Y-m-d", strtotime("+4 days"));

        $clients = $this->oModele->getListeClients(); // 👈 crée cette méthode

        $this->oVue->afficherFormValidation([
            'articles' => $articles,
            'totalHT' => $totalHT,
            'tva' => $tva,
            'dateCommande' => $dateCommande,
            'dateLivraison' => $dateLivraison,
            'clients' => $clients,
            'vendeur' => $_SESSION['login']
        ]);
    }

    public function valider_commande()
    {
        if (!isset($_POST['codec'], $_POST['date_commande'], $_POST['date_livraison']) || !isset($_SESSION['panier_json'])) {
            header("Location: index.php?gestion=commande");
            exit;
        }

        $client   = $_POST['codec'];
        $dateCmd  = date('Y-m-d H:i:s');
        $dateLiv  = $_POST['date_livraison'];
        $login    = $_SESSION['login'];
        $codev    = $this->oModele->getCodeVendeurByLogin($login);
        $panier   = json_decode($_SESSION['panier_json'], true);

        $totalHT  = 0;
        $totalTVA = 0;

        $articles = [];
        foreach ($panier as $ref => $qte) {
            $produit = $this->oModele->getProduitByReference($ref);
            if ($produit) {
                $prixHT      = $produit['prix_unitaire_HT'];
                $totalLigne  = $prixHT * $qte;
                $totalHT    += $totalLigne;
                $totalTVA   += ($prixHT * 0.2) * $qte;

                $articles[] = [
                    'reference' => $ref,
                    'quantite'  => $qte,
                    'prixHT'    => $prixHT
                ];
            }
        }

        // ➕ Source (booléen) pour la BDD : 0=Vendeur, 1=Client
        $src = $this->detectSourceBool();

        // ⬇ Insérer la commande (⚠️ adapter ton modèle pour accepter $src)
        $numeroCmd = $this->oModele->insererCommande($client, $codev, $dateCmd, $dateLiv, $totalHT, $totalTVA, $src);

        // ⬇ Insérer les lignes de commande
        $ligne = 1;
        foreach ($articles as $a) {
            $this->oModele->insererLigneCommande($numeroCmd, $ligne, $a['reference'], $a['quantite']);
            $this->oModele->mettreAJourStock($a['reference'], $a['quantite']);
            $ligne++;
        }

        unset($_SESSION['panier_json']);
        CommandeTable::setMessageSucces("Commande validée avec succès.");
        header("Location: http://localhost/sp-vendeurs/sp-vendeurs-11/index.php?gestion=commande");
        exit;
    }

}
